/*
 * Decompiled with CFR 0.152.
 */
package de.bmoth.cli;

import de.bmoth.modelchecker.ModelChecker;
import de.bmoth.modelchecker.ModelCheckingResult;
import de.bmoth.modelchecker.bmc.BoundedModelChecker;
import de.bmoth.modelchecker.esmc.ExplicitStateModelChecker;
import de.bmoth.modelchecker.kind.KInductionModelChecker;
import de.bmoth.parser.Parser;
import de.bmoth.parser.ParserException;
import de.bmoth.parser.ast.nodes.MachineNode;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.StringJoiner;
import java.util.logging.Logger;

public class CliTask {
    private static Logger logger = Logger.getAnonymousLogger();
    private boolean isBenchmark = false;
    private int maxSteps = 20;
    private int times = 1;
    private ModelCheckingAlgorithm algorithm = ModelCheckingAlgorithm.ESMC;
    private File machineFile = null;
    private String resultFileName = null;
    private long nanoDiffTime;
    private long parsingTimes = 0L;
    private long checkingTimes = 0L;

    CliTask() {
    }

    public void setAlgorithm(String algorithm) {
        this.algorithm = ModelCheckingAlgorithm.valueOf(algorithm.toUpperCase());
        logger.config("Setting algorithm to " + this.algorithm.verbose());
    }

    public void setIsBenchmark() {
        this.isBenchmark = true;
        logger.config("Enabling benchmark");
    }

    public void setMachineFile(File machineFile) {
        this.machineFile = machineFile;
        if (machineFile == null || !machineFile.exists()) {
            logger.warning("Setting invalid machine file");
        } else {
            logger.config("Setting machine file to " + machineFile.getAbsolutePath());
        }
    }

    public void setMaxSteps(int maxSteps) {
        this.maxSteps = maxSteps;
        logger.config("Setting max steps to " + maxSteps);
    }

    public void setTimes(int times) {
        this.times = times;
        logger.config("Setting times to " + times);
    }

    public void setResultFileName(String resultFileName) {
        this.resultFileName = resultFileName;
        logger.config("Setting resultFileName to " + resultFileName);
    }

    public void executeBenchmarks() {
        if (this.times != 0) {
            MachineNode machineNode = this.parseMachine(this.readMachineContent());
            String result = this.doModelCheck(this.getModelChecker(machineNode)).toString();
            for (int i = 1; i <= this.times; ++i) {
                logger.info("Executing benchmark " + i + " of " + this.times);
                machineNode = this.parseMachine(this.readMachineContent());
                this.parsingTimes += this.nanoDiffTime;
                result = this.doModelCheck(this.getModelChecker(machineNode)).toString();
                this.checkingTimes += this.nanoDiffTime;
            }
            StringJoiner resultString = new StringJoiner("\n", "", "\n\n");
            resultString.add(LocalDateTime.now().format(DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss")));
            resultString.add("Machine: " + this.machineFile.getName() + ", algorithm: " + (Object)((Object)this.algorithm) + ", maxSteps: " + this.maxSteps + ", times: " + this.times);
            resultString.add("Result: " + result);
            resultString.add("Average parsing time: " + this.parsingTimes / (long)this.times + " ns");
            resultString.add("Average checking time: " + this.checkingTimes / (long)this.times + " ns");
            logger.info(resultString.toString());
            if (this.resultFileName != null) {
                try {
                    PrintWriter writer = new PrintWriter(new FileOutputStream(new File(this.resultFileName), true));
                    writer.append(resultString.toString());
                    writer.close();
                }
                catch (Exception e) {
                    logger.warning(e.toString());
                }
            }
        }
    }

    public void execute() {
        if (this.isBenchmark) {
            this.executeBenchmarks();
        } else {
            MachineNode machineNode = this.parseMachine(this.readMachineContent());
            ModelCheckingResult result = this.doModelCheck(this.getModelChecker(machineNode));
            logger.info("Result: " + result.toString());
        }
    }

    private ModelCheckingResult doModelCheck(ModelChecker modelChecker) {
        long start = this.isBenchmark ? System.nanoTime() : 0L;
        ModelCheckingResult result = modelChecker.check();
        this.nanoDiffTime = this.isBenchmark ? System.nanoTime() - start : 0L;
        return result;
    }

    private ModelChecker getModelChecker(MachineNode machineNode) {
        switch (this.algorithm) {
            case ESMC: {
                return new ExplicitStateModelChecker(machineNode);
            }
            case BMC: {
                return new BoundedModelChecker(machineNode, this.maxSteps);
            }
            case KIND: {
                return new KInductionModelChecker(machineNode, this.maxSteps);
            }
        }
        logger.severe("Unknown algorithm " + (Object)((Object)this.algorithm));
        System.exit(1);
        return null;
    }

    private MachineNode parseMachine(String machineContent) {
        MachineNode machineNode = null;
        try {
            long start = this.isBenchmark ? System.nanoTime() : 0L;
            machineNode = Parser.getMachineAsSemanticAst(machineContent);
            this.nanoDiffTime = this.isBenchmark ? System.nanoTime() - start : 0L;
        }
        catch (ParserException e) {
            logger.severe(e.toString());
            System.exit(1);
        }
        if (machineNode == null) {
            logger.severe("Invalid machine");
            System.exit(1);
        }
        return machineNode;
    }

    private String readMachineContent() {
        if (!(this.machineFile != null && this.machineFile.exists() && this.machineFile.isFile() && this.machineFile.canRead())) {
            logger.severe("Unable to read file " + this.machineFile);
            System.exit(1);
        }
        String machineContent = null;
        try {
            machineContent = new String(Files.readAllBytes(Paths.get(this.machineFile.getPath(), new String[0])));
        }
        catch (IOException e) {
            logger.severe(e.toString());
            System.exit(1);
        }
        if (machineContent == null || machineContent.isEmpty()) {
            logger.severe("Missing machine");
            System.exit(1);
        }
        return machineContent;
    }

    private static enum ModelCheckingAlgorithm {
        ESMC("Explicit-state model checking"),
        BMC("Bounded model checking"),
        KIND("k-induction model checking");

        private final String name;

        private ModelCheckingAlgorithm(String name) {
            this.name = name;
        }

        public String verbose() {
            return this.name;
        }
    }
}

